/*
 * Decompiled with CFR 0.152.
 */
package net.messagevortex.transport;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import net.messagevortex.MessageVortexLogger;
import net.messagevortex.transport.SecurityContext;
import net.messagevortex.transport.SecurityRequirement;
import net.messagevortex.transport.imap.ImapLine;

public abstract class AbstractConnection {
    protected static final String CRLF = "\r\n";
    private static final Logger LOGGER = MessageVortexLogger.getLogger(new Throwable().getStackTrace()[0].getClassName());
    private static long defaultTimeout = 10000L;
    private long timeout = defaultTimeout;
    private ByteBuffer outboundEncryptedData = ByteBuffer.allocate(1024).clear();
    private ByteBuffer inboundEncryptedData = ByteBuffer.allocate(1024).clear().flip();
    private ByteBuffer outboundAppData = ByteBuffer.allocate(1024).clear();
    private ByteBuffer inboundAppData = ByteBuffer.allocate(1024).clear().flip();
    private String protocol = null;
    private boolean isClient;
    private final ExecutorService executor = Executors.newSingleThreadExecutor();
    private SecurityContext context = null;
    private boolean isTls = false;
    private SocketChannel socketChannel = null;
    private InetSocketAddress remoteAddress = null;
    private SSLEngine engine = null;
    private volatile boolean shutdownAbstractConnection = false;

    public AbstractConnection(InetSocketAddress remoteAddress, SecurityContext context) {
        this.remoteAddress = remoteAddress;
        this.setSecurityContext(context);
    }

    public AbstractConnection(AbstractConnection ac) {
        if (ac != null) {
            this.setSecurityContext(ac.getSecurityContext());
            this.setEngine(ac.getEngine());
            this.isTls = ac.isTls;
            this.socketChannel = ac.socketChannel;
            this.remoteAddress = ac.remoteAddress;
            this.protocol = ac.protocol;
            this.isClient = ac.isClient;
            this.timeout = ac.timeout;
            this.outboundEncryptedData = ac.outboundEncryptedData;
            this.outboundAppData = ac.outboundAppData;
            this.inboundEncryptedData = ac.inboundEncryptedData;
            this.inboundAppData = ac.inboundAppData;
        }
    }

    public AbstractConnection(SocketChannel sock, SecurityContext context, boolean isClient) {
        this.isClient = isClient;
        if (sock != null) {
            this.setSocketChannel(sock);
        }
        this.setSecurityContext(context);
    }

    public AbstractConnection(SocketChannel sock, SecurityContext context) {
        this(sock, context, true);
    }

    public String getHostName() {
        if (this.remoteAddress == null) {
            return null;
        }
        return this.remoteAddress.getHostName();
    }

    public int getPort() {
        if (this.remoteAddress == null) {
            return -1;
        }
        return this.remoteAddress.getPort();
    }

    protected final SocketChannel setSocketChannel(SocketChannel s) {
        SocketChannel ret = this.socketChannel;
        this.socketChannel = s;
        if (s != null) {
            try {
                s.configureBlocking(false);
                this.remoteAddress = (InetSocketAddress)s.getRemoteAddress();
            }
            catch (IOException ioe) {
                LOGGER.log(Level.SEVERE, "got unexpected exception", ioe);
            }
        }
        return ret;
    }

    public SocketChannel getSocketChannel() throws IOException {
        if (this.socketChannel == null || !this.socketChannel.isConnected()) {
            throw new IOException("socket is unexpectedly not connected (" + String.valueOf(this.socketChannel) + ")");
        }
        return this.socketChannel;
    }

    public final SecurityContext setSecurityContext(SecurityContext context) {
        SecurityContext ret = this.context;
        this.context = context;
        return ret;
    }

    public SecurityContext getSecurityContext() {
        return this.context;
    }

    protected SSLEngine getEngine() {
        return this.engine;
    }

    protected final SSLEngine setEngine(SSLEngine engine) {
        SSLEngine ret = this.engine;
        this.engine = engine;
        return ret;
    }

    public void connect() throws IOException {
        if (this.socketChannel == null) {
            this.socketChannel = SocketChannel.open();
            this.socketChannel.configureBlocking(false);
            LOGGER.log(Level.INFO, "connecting socket channel to " + String.valueOf(this.remoteAddress));
        }
        this.socketChannel.connect(this.remoteAddress);
        this.isClient = true;
        while (!this.socketChannel.finishConnect()) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (!(this.isTls() || this.context == null || this.context.getRequirement() != SecurityRequirement.SSLTLS && this.context.getRequirement() != SecurityRequirement.UNTRUSTED_SSLTLS)) {
            this.startTls();
        } else if (this.context != null && (this.context.getRequirement() == SecurityRequirement.SSLTLS || this.context.getRequirement() == SecurityRequirement.UNTRUSTED_SSLTLS)) {
            throw new IOException("unable to start required handshake due to missing SSLContext");
        }
        if (!this.socketChannel.isConnected()) {
            throw new IOException("socket is unexpectedly not connected");
        }
    }

    public void startTls() throws IOException {
        this.startTls(this.getTimeout());
    }

    public void startTls(long timeout) throws IOException {
        if (this.isTls()) {
            LOGGER.log(Level.WARNING, "refused to renegotiate TLS (already encrypted)");
            return;
        }
        if (this.getEngine() == null && this.getSecurityContext() != null && this.getSecurityContext().getContext() != null) {
            InetSocketAddress ia = (InetSocketAddress)this.socketChannel.getLocalAddress();
            LOGGER.log(Level.FINE, "created SSLEngine for " + String.valueOf(ia) + " (name=" + ia.getHostName() + "; client=" + this.isClient + ")");
            this.setEngine(this.getSecurityContext().getContext().createSSLEngine(ia.getHostName(), ia.getPort()));
            this.getEngine().setUseClientMode(this.isClient);
        } else if (this.getEngine() == null) {
            throw new IOException("no security context available but startTls called");
        }
        LOGGER.log(Level.FINE, "starting TLS handshake (clientMode=" + this.isClient + ")");
        this.isTls = true;
        this.do_handshake(timeout);
    }

    public String setProtocol(String protocol) {
        String ret = this.getProtocol();
        this.protocol = protocol;
        return ret;
    }

    public String getProtocol() {
        return this.protocol;
    }

    protected void do_handshake(long timeout) throws IOException {
        if (this.getEngine() == null) {
            throw new IOException("No SSL context available");
        }
        if (!this.getSocketChannel().isConnected()) {
            throw new IOException("No SSL connection possible on unconnected socket");
        }
        this.getEngine().beginHandshake();
        this.processEngineRequirements(timeout);
    }

    private int readRawSocket(long timeout) throws IOException {
        boolean loopOnce = this.inboundEncryptedData.remaining() > 0;
        int bytesRead = 0;
        this.inboundEncryptedData.compact();
        try {
            LOGGER.log(Level.FINE, "starting reading raw socket (loopOnce is " + loopOnce + ")");
            long start = System.currentTimeMillis();
            while (!this.shutdownAbstractConnection && bytesRead == 0 && timeout - (System.currentTimeMillis() - start) > 0L) {
                bytesRead = this.getSocketChannel().read(this.inboundEncryptedData);
                LOGGER.log(Level.INFO, "tried to read (got " + bytesRead + ")");
                if (bytesRead != 0) continue;
                if (loopOnce) {
                    LOGGER.log(Level.INFO, "done reading raw socket (loop once gave no new input)");
                    break;
                }
                try {
                    Thread.sleep(50L);
                }
                catch (InterruptedException interruptedException) {}
            }
            LOGGER.log(Level.FINE, "done reading raw socket (took " + (System.currentTimeMillis() - start) + "ms; read " + bytesRead + " bytes)");
        }
        catch (IOException ioe) {
            this.inboundEncryptedData.flip();
            LOGGER.log(Level.WARNING, "got exception while reading (propagated)", ioe);
            throw ioe;
        }
        this.inboundEncryptedData.flip();
        LOGGER.log(Level.FINE, "reverting inbound buffer (new size is " + this.inboundEncryptedData.remaining() + ")");
        return bytesRead;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int writeRawSocket(long timeout) throws IOException {
        this.outboundEncryptedData.flip();
        LOGGER.log(Level.INFO, "trying to send outboundEncryptedData Buffer " + this.outboundEncryptedData.remaining());
        try {
            long start = System.currentTimeMillis();
            int bytesWritten = 0;
            while (this.outboundEncryptedData.hasRemaining() && timeout - (System.currentTimeMillis() - start) > 0L) {
                bytesWritten += this.getSocketChannel().write(this.outboundEncryptedData);
            }
            int n = bytesWritten;
            return n;
        }
        finally {
            LOGGER.log(Level.FINE, "sending outboundEncryptedData Buffer done (" + this.outboundEncryptedData.remaining() + ")");
            this.outboundEncryptedData.compact();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processEngineRequirements(long timeout) throws IOException {
        if (this.getEngine() == null) {
            throw new IOException("No SSL context available");
        }
        if (!this.getSocketChannel().isConnected()) {
            throw new IOException("No SSL connection possible on unconnected socket");
        }
        LOGGER.log(Level.FINER, "status (1) outboundEncryptedData Buffer " + String.valueOf(this.outboundEncryptedData));
        LOGGER.log(Level.FINER, "status (1) outboundAppData Buffer " + String.valueOf(this.outboundAppData));
        LOGGER.log(Level.FINER, "status (1) inboundEncryptedData Buffer " + String.valueOf(this.inboundEncryptedData));
        LOGGER.log(Level.FINER, "status (1) inboundAppData Buffer " + String.valueOf(this.inboundAppData));
        long start = System.currentTimeMillis();
        SSLEngineResult.HandshakeStatus handshakeStatus = this.getEngine().getHandshakeStatus();
        while (!this.shutdownAbstractConnection && handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED && handshakeStatus != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING && timeout - (System.currentTimeMillis() - start) > 0L && this.getEngine() != null) {
            LOGGER.log(Level.FINE, "re-looping (handshake status is " + String.valueOf((Object)handshakeStatus) + ")");
            block8 : switch (handshakeStatus) {
                case NEED_UNWRAP: {
                    SSLEngineResult res;
                    LOGGER.log(Level.FINE, "doing unwrap (reading; buffer is " + this.inboundEncryptedData.remaining() + ")");
                    int bytesRead = this.readRawSocket(timeout - (System.currentTimeMillis() - start));
                    LOGGER.log(Level.FINE, "doing unwrap (" + this.inboundEncryptedData.remaining() + ")");
                    if (bytesRead < 0) {
                        if (this.getEngine().isInboundDone() && this.getEngine().isOutboundDone()) {
                            throw new IOException("cannot starttls. Socket was at least partially closed or the SSL engine encountered an error while handshaking");
                        }
                        try {
                            this.getEngine().closeInbound();
                        }
                        catch (SSLException e) {
                            LOGGER.log(Level.WARNING, "Exception while closing inbound", e);
                        }
                        this.getEngine().closeOutbound();
                        handshakeStatus = this.getEngine().getHandshakeStatus();
                        break;
                    }
                    this.inboundAppData.compact();
                    try {
                        res = this.getEngine().unwrap(this.inboundEncryptedData, this.inboundAppData);
                    }
                    catch (SSLException sslException) {
                        LOGGER.log(Level.WARNING, "A problem was encountered while processing the data that caused the SSLEngine to abort. Will try to properly close connection...", sslException);
                        this.getEngine().closeOutbound();
                        handshakeStatus = this.getEngine().getHandshakeStatus();
                        break;
                    }
                    finally {
                        this.inboundAppData.flip();
                    }
                    handshakeStatus = this.getEngine().getHandshakeStatus();
                    switch (res.getStatus()) {
                        case OK: {
                            LOGGER.log(Level.FINE, "unwrap done (OK;inboundAppData=" + this.inboundAppData.remaining() + ";inboundEncryptedData=" + this.inboundEncryptedData.remaining() + ")");
                            break block8;
                        }
                        case CLOSED: {
                            if (this.getEngine().isOutboundDone()) {
                                throw new IOException("engine reports closed status");
                            }
                            this.getEngine().closeOutbound();
                            handshakeStatus = this.getEngine().getHandshakeStatus();
                            LOGGER.log(Level.FINE, "unwrap done (CLOSED)");
                            this.isTls = false;
                            this.setEngine(null);
                            break block8;
                        }
                        case BUFFER_UNDERFLOW: {
                            this.inboundEncryptedData = this.handleBufferUnderflow(this.getEngine(), this.inboundEncryptedData);
                            handshakeStatus = this.getEngine().getHandshakeStatus();
                            LOGGER.log(Level.FINE, "unwrap done (UNDERFLOW; inboundEncryptedData Buffer " + this.inboundEncryptedData.remaining() + "/" + this.inboundEncryptedData.capacity() + ")");
                            break block8;
                        }
                        case BUFFER_OVERFLOW: {
                            this.inboundAppData = this.enlargeApplicationBuffer(this.getEngine(), this.inboundAppData);
                            handshakeStatus = this.getEngine().getHandshakeStatus();
                            LOGGER.log(Level.FINE, "unwrap done (OVERFLOW; inboundAppData Buffer " + this.inboundAppData.remaining() + "/" + this.inboundAppData.capacity() + ")");
                            break block8;
                        }
                    }
                    throw new IllegalStateException("Invalid SSL status: " + String.valueOf((Object)res.getStatus()));
                }
                case NEED_WRAP: {
                    LOGGER.log(Level.FINE, "doing wrap (wrapping)");
                    this.outboundAppData.flip();
                    SSLEngineResult result = null;
                    try {
                        LOGGER.log(Level.FINER, "outboundAppData (flipped)=" + String.valueOf(this.outboundAppData));
                        LOGGER.log(Level.FINER, "outboundEncryptedData=" + String.valueOf(this.outboundEncryptedData));
                        result = this.getEngine().wrap(this.outboundAppData, this.outboundEncryptedData);
                        handshakeStatus = result.getHandshakeStatus();
                        LOGGER.log(Level.FINER, "outboundAppData (flipped)=" + String.valueOf(this.outboundAppData));
                        LOGGER.log(Level.FINER, "outboundEncryptedData=" + String.valueOf(this.outboundEncryptedData) + "/" + String.valueOf(result));
                    }
                    catch (SSLException se) {
                        LOGGER.log(Level.WARNING, "Exception while reading data", se);
                        this.getEngine().closeOutbound();
                        handshakeStatus = this.getEngine().getHandshakeStatus();
                        break;
                    }
                    finally {
                        this.outboundAppData.compact();
                    }
                    this.outboundEncryptedData.flip();
                    LOGGER.log(Level.FINE, "doing wrap (outboundEncryptedData is " + this.outboundEncryptedData.remaining() + "/" + this.outboundEncryptedData.capacity() + ")");
                    this.outboundEncryptedData.compact();
                    switch (result.getStatus()) {
                        case OK: {
                            this.writeRawSocket(timeout - (System.currentTimeMillis() - start));
                            this.outboundEncryptedData.flip();
                            LOGGER.log(Level.FINE, "wrap done with status OK (remaining bytes:" + this.outboundEncryptedData.remaining() + ")");
                            this.outboundEncryptedData.compact();
                            handshakeStatus = result.getHandshakeStatus();
                            break block8;
                        }
                        case CLOSED: {
                            try {
                                this.writeRawSocket(timeout - (System.currentTimeMillis() - start));
                            }
                            catch (IOException e) {
                                LOGGER.log(Level.FINE, "Failed to close channel", e);
                            }
                            this.isTls = false;
                            this.setEngine(null);
                            handshakeStatus = result.getHandshakeStatus();
                            break block8;
                        }
                        case BUFFER_UNDERFLOW: {
                            LOGGER.log(Level.SEVERE, "Buffer underflow should not happen", new SSLException("unknown reason for buffer underflow"));
                            handshakeStatus = result.getHandshakeStatus();
                            break block8;
                        }
                        case BUFFER_OVERFLOW: {
                            this.outboundEncryptedData.flip();
                            this.outboundEncryptedData = this.enlargePacketBuffer(this.getEngine(), this.outboundEncryptedData);
                            LOGGER.log(Level.FINE, "wrap done (OVERFLOW; outboundEncryptedData Buffer " + this.outboundEncryptedData.remaining() + "/" + this.outboundEncryptedData.capacity() + ")");
                            this.outboundEncryptedData.compact();
                            handshakeStatus = result.getHandshakeStatus();
                            break block8;
                        }
                    }
                    throw new IllegalStateException("Invalid SSL status: " + String.valueOf((Object)result.getStatus()));
                }
                case NEED_TASK: {
                    LOGGER.log(Level.FINE, "running tasks");
                    Runnable task = this.getEngine().getDelegatedTask();
                    do {
                        if (task != null) {
                            LOGGER.log(Level.FINE, "running task " + String.valueOf(task));
                            Thread t = new Thread(task);
                            t.start();
                            try {
                                t.join();
                            }
                            catch (InterruptedException interruptedException) {}
                            continue;
                        }
                        try {
                            Thread.sleep(20L);
                        }
                        catch (InterruptedException interruptedException) {
                            // empty catch block
                        }
                    } while ((task = this.getEngine().getDelegatedTask()) != null);
                    LOGGER.log(Level.FINE, "running tasks done");
                    handshakeStatus = this.getEngine().getHandshakeStatus();
                    break;
                }
                case FINISHED: {
                    LOGGER.log(Level.FINE, "TLS handshake success");
                    break;
                }
                case NOT_HANDSHAKING: {
                    break;
                }
                default: {
                    throw new IllegalStateException("Invalid SSL status: " + String.valueOf((Object)handshakeStatus));
                }
            }
            this.outboundEncryptedData.flip();
            LOGGER.log(Level.FINE, "status outboundEncryptedData Buffer " + this.outboundEncryptedData.remaining() + "/" + this.outboundEncryptedData.capacity());
            this.outboundEncryptedData.compact();
            this.outboundAppData.flip();
            LOGGER.log(Level.FINE, "status outboundAppData Buffer " + this.outboundAppData.remaining() + "/" + this.outboundAppData.capacity());
            this.outboundAppData.compact();
            LOGGER.log(Level.FINE, "status inboundEncryptedData Buffer " + this.inboundEncryptedData.remaining() + "/" + this.inboundEncryptedData.capacity());
            LOGGER.log(Level.FINE, "status inboundAppData Buffer " + this.inboundAppData.remaining() + "/" + this.inboundAppData.capacity());
        }
        long elapsed = timeout - (System.currentTimeMillis() - start);
        if (!this.shutdownAbstractConnection && elapsed <= 0L && handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
            throw new IOException("SSL/TLS handshake aborted due to timeout (current timeout:" + timeout / 1000L + "s ;elapsed:" + elapsed / 1000L + "s)");
        }
        LOGGER.log(Level.FINE, "******* HANDSHAKE SUCCESS");
    }

    protected void do_teardown(long timeout) throws IOException {
        long start = System.currentTimeMillis();
        if (this.isTls()) {
            LOGGER.log(Level.FINE, "starting TLS teardown");
            this.getEngine().closeOutbound();
            this.outboundAppData.clear();
            while (!this.getEngine().isOutboundDone()) {
                this.outboundAppData.flip();
                SSLEngineResult res = this.getEngine().wrap(this.outboundAppData, this.outboundEncryptedData);
                this.outboundAppData.compact();
                switch (res.getStatus()) {
                    case OK: {
                        break;
                    }
                    case CLOSED: {
                        break;
                    }
                    case BUFFER_UNDERFLOW: {
                        this.outboundAppData.flip();
                        this.outboundAppData = this.handleBufferUnderflow(this.getEngine(), this.outboundAppData);
                        this.outboundAppData.compact();
                        break;
                    }
                    case BUFFER_OVERFLOW: {
                        this.outboundEncryptedData.flip();
                        this.outboundEncryptedData = this.enlargePacketBuffer(this.getEngine(), this.outboundEncryptedData);
                        this.outboundEncryptedData.compact();
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Invalid SSL status: " + String.valueOf((Object)res.getStatus()));
                    }
                }
                this.writeRawSocket(timeout - (System.currentTimeMillis() - start));
            }
        }
        LOGGER.log(Level.FINE, "TLS teardown done");
    }

    public boolean isTls() {
        return this.isTls;
    }

    public static long setDefaultTimeout(long timeout) {
        long ret = defaultTimeout;
        defaultTimeout = timeout;
        return ret;
    }

    public static long getDefaultTimeout() {
        return defaultTimeout;
    }

    public long setTimeout(long timeout) {
        long ret = this.timeout;
        this.timeout = timeout;
        return ret;
    }

    public long getTimeout() {
        return this.timeout;
    }

    private void writeSocket(String message, long timeout) throws IOException {
        int msgSize;
        long start = System.currentTimeMillis();
        int n = msgSize = message == null ? 0 : message.getBytes(StandardCharsets.UTF_8).length;
        if (this.outboundAppData.limit() < msgSize) {
            this.outboundAppData = this.enlargeBuffer(this.outboundAppData, this.outboundAppData.capacity() + msgSize);
        }
        if (message != null) {
            this.outboundAppData.put(message.getBytes(StandardCharsets.UTF_8));
        }
        this.outboundAppData.flip();
        while (!this.shutdownAbstractConnection && this.outboundAppData.remaining() > 0 && start + timeout > System.currentTimeMillis()) {
            this.outboundAppData.compact();
            if (this.isTls()) {
                this.outboundAppData.flip();
                SSLEngineResult result = this.getEngine().wrap(this.outboundAppData, this.outboundEncryptedData);
                this.outboundAppData.compact();
                switch (result.getStatus()) {
                    case OK: {
                        this.writeRawSocket(timeout - (System.currentTimeMillis() - start));
                        break;
                    }
                    case BUFFER_OVERFLOW: {
                        this.outboundAppData.flip();
                        this.outboundEncryptedData = this.enlargePacketBuffer(this.getEngine(), this.outboundEncryptedData);
                        this.outboundAppData.compact();
                        break;
                    }
                    case BUFFER_UNDERFLOW: {
                        throw new SSLException("Buffer underflow occurred after a wrap. I don't think we should ever get here.");
                    }
                    case CLOSED: {
                        this.closeConnection();
                        return;
                    }
                    default: {
                        throw new IllegalStateException("Invalid SSL status: " + String.valueOf((Object)result.getStatus()));
                    }
                }
            } else {
                this.outboundAppData.flip();
                this.outboundEncryptedData.put(this.outboundAppData);
                this.outboundAppData.compact();
                this.writeRawSocket(timeout - (System.currentTimeMillis() - start));
            }
            this.outboundAppData.flip();
        }
        this.outboundAppData.compact();
        if (start + timeout <= System.currentTimeMillis()) {
            throw new IOException("Timeout reached while writing");
        }
    }

    private int readSocket(long timeout) throws IOException, TimeoutException {
        boolean timeoutReached;
        int bytesRead;
        int totBytesRead = 0;
        long start = System.currentTimeMillis();
        do {
            try {
                bytesRead = this.readRawSocket(timeout - (System.currentTimeMillis() - start));
            }
            catch (IOException ioe) {
                bytesRead = 0;
                if (timeout <= System.currentTimeMillis() - start) {
                    LOGGER.log(Level.FINE, "timeout has been reached while waiting for input");
                    TimeoutException e = new TimeoutException("Timeout while reading");
                    e.addSuppressed(ioe);
                    throw e;
                }
                throw ioe;
            }
            if (bytesRead == 0) {
                LOGGER.log(Level.FINE, "sleeping due to missing data (" + this.inboundEncryptedData.remaining() + ")");
                try {
                    Thread.sleep(50L);
                }
                catch (InterruptedException ioe) {
                    // empty catch block
                }
            }
            if (this.inboundEncryptedData.remaining() > 0) {
                if (this.isTls()) {
                    bytesRead = -this.inboundAppData.remaining();
                    this.inboundAppData.compact();
                    LOGGER.log(Level.FINE, "decrypting (occupied inbound buffer space: " + this.inboundEncryptedData.remaining() + "; counter: " + bytesRead + ")", new Object[]{this.inboundEncryptedData, this.inboundAppData});
                    SSLEngineResult result = this.getEngine().unwrap(this.inboundEncryptedData, this.inboundAppData);
                    this.inboundAppData.flip();
                    LOGGER.log(Level.FINE, "decryption done (occupied buffer space: " + this.inboundAppData.remaining() + "; counter: " + (bytesRead += this.inboundAppData.remaining()) + ")");
                    if (this.getEngine().getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED && this.getEngine().getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
                        LOGGER.log(Level.FINE, "Handshake status is " + String.valueOf((Object)this.getEngine().getHandshakeStatus()));
                        this.processEngineRequirements(timeout - (System.currentTimeMillis() - start));
                    }
                    switch (result.getStatus()) {
                        case OK: {
                            break;
                        }
                        case BUFFER_OVERFLOW: {
                            this.inboundAppData = this.enlargeApplicationBuffer(this.getEngine(), this.inboundAppData);
                            break;
                        }
                        case BUFFER_UNDERFLOW: {
                            this.inboundEncryptedData = this.handleBufferUnderflow(this.getEngine(), this.inboundEncryptedData);
                            break;
                        }
                        case CLOSED: {
                            this.closeConnection();
                            break;
                        }
                        default: {
                            throw new IllegalStateException("Invalid SSL status: " + String.valueOf((Object)result.getStatus()));
                        }
                    }
                } else {
                    this.inboundAppData.compact();
                    if (this.inboundAppData.limit() < bytesRead) {
                        this.inboundAppData.flip();
                        this.inboundAppData = this.enlargeBuffer(this.inboundAppData, this.inboundEncryptedData.remaining() + this.inboundAppData.capacity());
                        this.inboundAppData.compact();
                    }
                    this.inboundAppData.put(this.inboundEncryptedData);
                    this.inboundAppData.flip();
                }
                if (bytesRead > 0) {
                    totBytesRead += bytesRead;
                    LOGGER.log(Level.FINE, "got bytes (" + bytesRead + " bytes; occupied buffer space: " + this.inboundEncryptedData.remaining() + ")");
                }
            }
            boolean bl = timeoutReached = System.currentTimeMillis() - start > timeout;
        } while (!this.shutdownAbstractConnection && bytesRead >= 0 && totBytesRead == 0 && !timeoutReached && timeout > 0L);
        if (bytesRead < 0) {
            LOGGER.log(Level.FINE, "Loop aborted due to closed connection");
        } else if (System.currentTimeMillis() - start > timeout) {
            LOGGER.log(Level.FINE, "Loop aborted due to timeout");
        } else if (timeout <= 0L) {
            LOGGER.log(Level.INFO, "Loop aborted due to single round wish");
        } else if (totBytesRead > 0) {
            LOGGER.log(Level.FINE, "Loop aborted due to data (" + totBytesRead + " bytes; occupied buffer space: " + this.inboundAppData.remaining() + ")");
        } else {
            LOGGER.log(Level.FINE, "Loop aborted due to UNKNOWN");
        }
        if (bytesRead < 0) {
            LOGGER.log(Level.FINE, "doing shutdown due to closed connection");
            this.shutdown();
        }
        return totBytesRead;
    }

    public void writeln(String message) throws IOException {
        this.writeln(message, this.getTimeout());
    }

    public void writeln(String message, long timeout) throws IOException {
        this.write(message + CRLF, timeout);
    }

    public void write(String message) throws IOException {
        this.write(message, this.getTimeout());
    }

    public void write(String message, long timeout) throws IOException {
        LOGGER.log(Level.FINE, "writing message with write (" + ImapLine.commandEncoder(message) + "; size " + (message != null ? message.length() : -1) + ")");
        this.writeSocket(message, this.getTimeout());
    }

    public String read() throws IOException, TimeoutException {
        return this.read(this.getTimeout());
    }

    public String read(long timeout) throws IOException, TimeoutException {
        int numBytes = this.readSocket(timeout);
        if (numBytes > 0) {
            byte[] b = new byte[numBytes];
            this.inboundAppData.get(b);
            this.inboundAppData.compact().flip();
            return new String(b, StandardCharsets.UTF_8);
        }
        return "";
    }

    public String readln() throws IOException, TimeoutException {
        return this.readln(this.getTimeout());
    }

    public String readln(long timeout) throws IOException, TimeoutException {
        StringBuilder ret = new StringBuilder();
        long start = System.currentTimeMillis();
        try {
            while (!this.shutdownAbstractConnection && !ret.toString().endsWith(CRLF) && timeout - (System.currentTimeMillis() - start) > 0L) {
                if (!this.inboundAppData.hasRemaining()) {
                    this.readSocket(timeout - (System.currentTimeMillis() - start));
                }
                if (this.inboundAppData.hasRemaining()) {
                    ret.append((char)this.inboundAppData.get());
                    continue;
                }
                try {
                    Thread.sleep(40L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        catch (IOException ioe) {
            if (timeout <= System.currentTimeMillis() - start) {
                LOGGER.log(Level.INFO, "timeout has been reached while waiting for input");
            }
            throw ioe;
        }
        if (!ret.toString().endsWith(CRLF)) {
            this.inboundAppData.rewind();
            this.inboundAppData.compact().flip();
            return null;
        }
        LOGGER.log(Level.FINE, "got line from readline (" + ret.substring(0, ret.length() - 2) + "; size " + ret.length() + ")");
        this.inboundAppData.compact().flip();
        return ret.substring(0, ret.length() - 2);
    }

    protected ByteBuffer enlargePacketBuffer(SSLEngine engine, ByteBuffer buffer) {
        return this.enlargeBuffer(buffer, engine.getSession().getPacketBufferSize());
    }

    protected ByteBuffer enlargeApplicationBuffer(SSLEngine engine, ByteBuffer buffer) {
        return this.enlargeBuffer(buffer, engine.getSession().getApplicationBufferSize());
    }

    protected ByteBuffer enlargeBuffer(ByteBuffer buffer, int sessionProposedCapacity) {
        ByteBuffer newBuffer = sessionProposedCapacity > buffer.capacity() ? ByteBuffer.allocate(sessionProposedCapacity) : ByteBuffer.allocate(buffer.capacity() * 2);
        newBuffer.put(buffer);
        newBuffer.flip();
        return newBuffer;
    }

    protected ByteBuffer handleBufferUnderflow(SSLEngine engine, ByteBuffer buffer) {
        if (engine.getSession().getPacketBufferSize() <= buffer.limit()) {
            return buffer;
        }
        return this.enlargePacketBuffer(engine, buffer);
    }

    protected void closeConnection() throws IOException {
        if (this.getEngine() != null && this.isTls()) {
            this.do_teardown(1000L);
        }
        if (this.socketChannel != null) {
            this.socketChannel.close();
        }
    }

    protected void handleEndOfStream() throws IOException {
        if (this.getEngine() != null) {
            try {
                this.getEngine().closeInbound();
            }
            catch (IOException ioe) {
                LOGGER.log(Level.WARNING, "channel already closed without TLS closure", ioe);
            }
        }
        this.closeConnection();
    }

    public void shutdown() throws IOException {
        this.shutdownAbstractConnection = true;
        this.executor.shutdown();
    }

    public boolean isShutdown() {
        return this.shutdownAbstractConnection;
    }
}

